﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/connect/ConnectRequest.h>
#include <aws/connect/Connect_EXPORTS.h>
#include <aws/connect/model/DirectoryType.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Connect {
namespace Model {

/**
 */
class CreateInstanceRequest : public ConnectRequest {
 public:
  AWS_CONNECT_API CreateInstanceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateInstance"; }

  AWS_CONNECT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The idempotency token.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateInstanceRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of identity management for your Amazon Connect users.</p>
   */
  inline DirectoryType GetIdentityManagementType() const { return m_identityManagementType; }
  inline bool IdentityManagementTypeHasBeenSet() const { return m_identityManagementTypeHasBeenSet; }
  inline void SetIdentityManagementType(DirectoryType value) {
    m_identityManagementTypeHasBeenSet = true;
    m_identityManagementType = value;
  }
  inline CreateInstanceRequest& WithIdentityManagementType(DirectoryType value) {
    SetIdentityManagementType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name for your instance.</p>
   */
  inline const Aws::String& GetInstanceAlias() const { return m_instanceAlias; }
  inline bool InstanceAliasHasBeenSet() const { return m_instanceAliasHasBeenSet; }
  template <typename InstanceAliasT = Aws::String>
  void SetInstanceAlias(InstanceAliasT&& value) {
    m_instanceAliasHasBeenSet = true;
    m_instanceAlias = std::forward<InstanceAliasT>(value);
  }
  template <typename InstanceAliasT = Aws::String>
  CreateInstanceRequest& WithInstanceAlias(InstanceAliasT&& value) {
    SetInstanceAlias(std::forward<InstanceAliasT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier for the directory.</p>
   */
  inline const Aws::String& GetDirectoryId() const { return m_directoryId; }
  inline bool DirectoryIdHasBeenSet() const { return m_directoryIdHasBeenSet; }
  template <typename DirectoryIdT = Aws::String>
  void SetDirectoryId(DirectoryIdT&& value) {
    m_directoryIdHasBeenSet = true;
    m_directoryId = std::forward<DirectoryIdT>(value);
  }
  template <typename DirectoryIdT = Aws::String>
  CreateInstanceRequest& WithDirectoryId(DirectoryIdT&& value) {
    SetDirectoryId(std::forward<DirectoryIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Your contact center handles incoming contacts.</p>
   */
  inline bool GetInboundCallsEnabled() const { return m_inboundCallsEnabled; }
  inline bool InboundCallsEnabledHasBeenSet() const { return m_inboundCallsEnabledHasBeenSet; }
  inline void SetInboundCallsEnabled(bool value) {
    m_inboundCallsEnabledHasBeenSet = true;
    m_inboundCallsEnabled = value;
  }
  inline CreateInstanceRequest& WithInboundCallsEnabled(bool value) {
    SetInboundCallsEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Your contact center allows outbound calls.</p>
   */
  inline bool GetOutboundCallsEnabled() const { return m_outboundCallsEnabled; }
  inline bool OutboundCallsEnabledHasBeenSet() const { return m_outboundCallsEnabledHasBeenSet; }
  inline void SetOutboundCallsEnabled(bool value) {
    m_outboundCallsEnabledHasBeenSet = true;
    m_outboundCallsEnabled = value;
  }
  inline CreateInstanceRequest& WithOutboundCallsEnabled(bool value) {
    SetOutboundCallsEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags used to organize, track, or control access for this resource. For
   * example, <code>{ "tags": {"key1":"value1", "key2":"value2"} }</code>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateInstanceRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateInstanceRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  DirectoryType m_identityManagementType{DirectoryType::NOT_SET};

  Aws::String m_instanceAlias;

  Aws::String m_directoryId;

  bool m_inboundCallsEnabled{false};

  bool m_outboundCallsEnabled{false};

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_clientTokenHasBeenSet = true;
  bool m_identityManagementTypeHasBeenSet = false;
  bool m_instanceAliasHasBeenSet = false;
  bool m_directoryIdHasBeenSet = false;
  bool m_inboundCallsEnabledHasBeenSet = false;
  bool m_outboundCallsEnabledHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Connect
}  // namespace Aws
