# Copyright (C) 2025 Intel Corporation
# SPDX-License-Identifier: MIT

import glob
import os
import re
import platform
from pathlib import Path


MAX_FILE_NAME_SIZE = 2000
MAX_FILE_SIZE_GB = 10
MAX_FILE_SIZE_BYTES = MAX_FILE_SIZE_GB * 1024 * 1024 * 1024


class StringValidator:
    def __init__(self, max_string_size=0):
        self.__max_string_size = max_string_size if max_string_size else MAX_FILE_NAME_SIZE

    def __call__(self, string) -> str:
        self._validate_string_length(string)
        self._validate_string_does_not_contain_null_characters(string)
        return string

    def _validate_string_length(self, string):
        if len(string) > self.__max_string_size:
            raise ValueError(f'strings cannot be longer than {self.__max_string_size} characters')

    @staticmethod
    def _validate_string_does_not_contain_null_characters(string):
        string_bytes = [b for b in string.encode()]
        if string_bytes.count(0) > 0:
            raise ValueError(f'String {string} contains invalid symbols')


class PathNameValidator(StringValidator):
    INVALID_CHARACTERS = r'[<>"|?*]'
    WINDOWS_RESERVED_NAMES = {'CON', 'PRN', 'AUX', 'NUL', 'COM1', 'COM2', 'COM3', 'COM4', 'COM5', 'COM6', 'COM7', 'COM8', 'COM9', 'LPT1', 'LPT2', 'LPT3', 'LPT4', 'LPT5', 'LPT6', 'LPT7', 'LPT8', 'LPT9'}

    def __init__(self, max_string_size=None):
        super().__init__(max_string_size)

    def __call__(self, string):
        super().__call__(string)
        self._validate_no_invalid_characters(string)
        self._validate_not_reserved_name(string)
        self._validate_colon_usage(string)
        return string

    def _validate_no_invalid_characters(self, string):
        if re.search(self.INVALID_CHARACTERS, string):
            raise ValueError(f'String {string} contains invalid characters')

    def _validate_not_reserved_name(self, string):
        if string.upper() in self.WINDOWS_RESERVED_NAMES and platform.system() == 'Windows':
            raise ValueError(f'String {string} is a reserved name')

    @staticmethod
    def _validate_colon_usage(string):
        if ':' in string:
            if platform.system() != 'Windows' or not re.match(r'^[a-zA-Z]:', string):
                raise ValueError(f'Invalid use of colon in path: {string}')


class FileValidator:
    """
    Validate general file attributes (existence, size)
    """

    def __init__(self, file_must_exist=True, file_must_not_exist=False, max_file_size=0, allow_symlinks=True):
        """
        Configure the file validator

        :param file_must_exist: when True, verifies that the specified file exists
        :param file_must_not_exist: when True, verifies that the specified file does not exist
        :param max_file_size: maximum file size allowed (0 means MAX_FILE_SIZE_BYTES)
        :param allow_symlinks: when True, allow using symlinks for existing files, otherwise reject symlinks
        """
        self.__must_exist = file_must_exist
        self.__must_not_exist = file_must_not_exist
        self.__max_file_size = max_file_size if max_file_size else MAX_FILE_SIZE_BYTES
        self.__validate_preconditions()
        self.__allow_symlinks = allow_symlinks

    def __call__(self, file_spec):
        """
        Verify the file

        :param file_spec: file to verify. Wildcards (glob expressions) are supported
        :return: upon successful verification, returns the absolute path of path_spec
        :raise ValueError: path_spec verification failed
        """
        self._validate_path_length(file_spec)
        self._validate_path_does_not_contain_null_characters(file_spec)

        if file_spec.startswith('"') and file_spec.endswith('"'):
            file_spec = file_spec.strip('"')
        absolute_path = os.path.abspath(file_spec)
        self._validate_path_format(absolute_path)
        if not self.__allow_symlinks and Path(absolute_path).is_symlink():
            raise ValueError(f'symbolic links not allowed: {absolute_path}')
        if self.__must_not_exist:
            self._validate_file_does_not_exist(absolute_path)
        if self.__must_exist:
            self._validate_file_exists(absolute_path)
        return absolute_path

    def __validate_preconditions(self):
        if self.__must_exist and self.__must_not_exist:
            raise ValueError('file_must_exist and file_must_not_exist cannot both be True')
        if self.__max_file_size and self.__max_file_size < 0:
            raise ValueError('max_file_size must be greater or equal to 0')

    def _validate_path_format(self, file_path):
        self._validate_parent_path(file_path)

    def _validate_file_exists(self, absolute_path):
        list_of_files = glob.glob(str(absolute_path))
        if len(list_of_files) == 0:
            raise ValueError(f'cannot find file(s): {absolute_path}')
        for file_path in list_of_files:
            self._validate_file_path(file_path)

    @staticmethod
    def _validate_file_does_not_exist(file_path):
        if os.path.isfile(file_path):
            raise ValueError(f'file already exists: {file_path}')
        # Verify file can be opened for writing
        try:
            with open(file_path, 'w'):
                pass
            os.remove(file_path)
        except Exception:
            raise ValueError(f'unable to write to file. Check file name and permissions: "{file_path}"')

    @staticmethod
    def _validate_parent_path(file_path):
        parent_dir = os.path.dirname(file_path)
        if parent_dir == '':
            parent_dir = '../../../mpp/core'
        if not os.path.isdir(parent_dir):
            raise ValueError(f'directory does not exist: {parent_dir}')

    @staticmethod
    def _validate_path_length(file_path):
        if len(file_path) > MAX_FILE_NAME_SIZE:
            raise ValueError(f'file names cannot be longer than {MAX_FILE_NAME_SIZE} characters')

    def _validate_file_size(self, file_path):
        if os.path.getsize(file_path) > self.__max_file_size:
            raise ValueError(f'input file size cannot be larger than '
                             f'{self.__max_file_size} bytes: {file_path}')

    def _validate_file_path(self, file_path):
        self._validate_path_length(file_path)
        if not os.path.isfile(file_path):
            raise ValueError(f'cannot find file(s): {file_path}')
        self._validate_file_size(file_path)
        return file_path

    @staticmethod
    def _validate_path_does_not_contain_null_characters(file_spec):
        file_spec_bytes = [b for b in file_spec.encode()]
        if file_spec_bytes.count(0) > 0:
            raise ValueError(f'illegal path name: {file_spec}')


class DirectoryValidator(FileValidator):

    class DirectoryValidatorError(ValueError):
        def __init__(self, absolute_path):
            super().__init__(f'directory does not exist: {absolute_path}')

    def __init__(self):
        super().__init__(True, False, allow_symlinks=True)

    def _validate_path_format(self, absolute_path):
        self._validate_directory_path(absolute_path)

    def _validate_directory_path(self, absolute_path):
        if not os.path.isdir(absolute_path):
            raise self.DirectoryValidatorError(absolute_path)

    def _validate_file_exists(self, absolute_path):
        self._validate_directory_path(absolute_path)


class ExcelFileValidator(FileValidator):

    class ExcelFileValidatorError(ValueError):
        def __init__(self, file_path):
            super().__init__(f'{file_path} is not a valid excel file name. A \'.xlsx\' extension is required')

    def __init__(self, file_must_exist=False, file_must_not_exist=False, max_file_size=0, allow_symlinks=True):
        super().__init__(file_must_exist, file_must_not_exist, max_file_size, allow_symlinks)

    def _validate_path_format(self, file_path):
        self._validate_xlsx(file_path)

    def _validate_xlsx(self, file_path: str):
        if Path(file_path).suffix != '.xlsx':
            raise self.ExcelFileValidatorError(file_path)

