# coding: utf-8
# typed: strict
# frozen_string_literal: true

require 'digest'
require 'openssl'

class PDF::Reader

  # Decrypts data using the AESV3 algorithim defined in the PDF 1.7, Extension Level 3 spec.
  # Requires a decryption key, which is usually generated by PDF::Reader::KeyBuilderV5
  #
  class AesV3SecurityHandler

    #: (String) -> void
    def initialize(key)
      if key.bytesize != 32
        raise PDF::Reader::MalformedPDFError.new(
          "AES-256 key must be exactly 32 bytes, got #{key.bytesize}"
        )
      end
      @encrypt_key = key
      @cipher = "AES-256-CBC" #: String
    end

    ##7.6.2 General Encryption Algorithm
    #
    # Algorithm 1: Encryption of data using the RC4 or AES algorithms
    #
    # used to decrypt RC4/AES encrypted PDF streams (buf). Input data should be in bytesizes of
    # a multiple of 16, anything else is an error. The first 16 bytes are the initialization
    # vector, so any input of exactly 16 bytes decrypts to an empty string
    #
    # buf - a string to decrypt
    # ref - a PDF::Reader::Reference for the object to decrypt
    #
    #: (String, PDF::Reader::Reference) -> String
    def decrypt( buf, ref )
      if buf.bytesize % 16 > 0
        raise PDF::Reader::MalformedPDFError.new("Ciphertext not a multiple of 16")
      elsif buf.bytesize == 16
        return ""
      else
        begin
          internal_decrypt(buf, ref)
        rescue OpenSSL::Cipher::CipherError
          # If we failed to decrypt it might be a padding error, so try again
          # and assume no padding in the ciphertext. This will "suceed" but might
          # return garbage if the key is incorrect but that's OK - well before this
          # class is used we have confirmed the user provided key is correct so if
          # this works without error we can be confident the returned plaintext is
          #  correct
         internal_decrypt(buf, ref, false)
        end
      end
    end

    private

    #: (String, PDF::Reader::Reference, ?bool) -> String
    def internal_decrypt(buf, ref, padding = true)
      cipher = OpenSSL::Cipher.new(@cipher)
      cipher.decrypt
      cipher.padding = 0 unless padding
      cipher.key = @encrypt_key.dup
      cipher.iv = buf[0..15]
      cipher.update(buf[16..-1]) + cipher.final
    end
  end
end
